<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\BankAccount;
use App\Models\BankAccountLedger;

class BankAccountController extends Controller
{
    public function index(Request $request)
    {
        $query = BankAccount::query();

        if ($search = $request->get('search')) {
            $query->where('account_name', 'like', "%$search%")
                ->orWhere('account_number', 'like', "%$search%")
                ->orWhere('ifsc_code', 'like', "%$search%");
        }

        $bankAccounts = $query->orderBy('id', 'desc')->paginate(10)->through(function ($inv) {
            return [
                'id' => $inv->id,
                'account_name'      => $inv->account_name,
                'account_number'    => $inv->account_number,
                'bank_name'         => $inv->bank_name,
                'branch_name'       => $inv->branch_name,
                'ifsc_code'         => $inv->ifsc_code,
                'balance'           => $inv->balance,
            ];
        });

        return inertia('App Settings/BankAccount/Index', [
            'bankAccounts' => $bankAccounts,
        ]);
    } 
    
     public function create()
    {
        return Inertia::render('App Settings/BankAccount/Create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'account_name'    => 'required|string|max:100',
            'account_number'  => 'required|string|max:100',
            'bank_name'       => 'required|string|max:100',
            'branch_name'     => 'required|string|max:100',
            'ifsc_code'       => 'required|string|max:100',
            'balance'         => 'required|numeric|min:0',
        ]);

        $bankAccount = BankAccount::create($data);

        $bankAccountLedger = new BankAccountLedger();

        $bankAccountLedger->bank_account_id = $bankAccount->id;
        $bankAccountLedger->transaction_date = date('Y-m-d');
        $bankAccountLedger->reference = "Account Opening Balance " . $request->balance;
        $bankAccountLedger->type = "debit";
        $bankAccountLedger->amount = $request->balance;
        $bankAccountLedger->carrier_id = 0;
        $bankAccountLedger->client_id = 0;
        // $bankAccountLedger->description = 'New Payme';

        $bankAccountLedger->save();

        return redirect()->route('bank-accounts.index')->with('success', 'Bank Account created successfully!');
    }

    public function show($id)
    {
        $bankAccount = BankAccount::findOrFail($id);

        return Inertia::render('App Settings/BankAccount/Show', [
            'bankAccount' => [
                'id'             => $bankAccount->id,
                'account_name'   => $bankAccount->account_name,
                'account_number' => $bankAccount->account_number,
                'bank_name'      => $bankAccount->bank_name,
                'branch_name'    => $bankAccount->branch_name,
                'ifsc_code'      => $bankAccount->ifsc_code,
                'balance'        => $bankAccount->balance,
            ],
        ]);
    }

    public function edit($id)
    {
        $bankAccount = BankAccount::findOrFail($id);

        return Inertia::render('App Settings/BankAccount/Edit', [
            'bankAccount' => [
                'id'             => $bankAccount->id,
                'account_name'   => $bankAccount->account_name,
                'account_number' => $bankAccount->account_number,
                'bank_name'      => $bankAccount->bank_name,
                'branch_name'    => $bankAccount->branch_name,
                'ifsc_code'      => $bankAccount->ifsc_code,
                'balance'        => $bankAccount->balance,
            ],
        ]);
    }   

    public function update(Request $request, $id)
    {
        $bankAccount = BankAccount::findOrFail($id);

        $data = $request->validate([
            'account_name'    => 'required|string|max:100',
            'account_number'  => 'required|string|max:100',
            'bank_name'       => 'required|string|max:100',
            'branch_name'     => 'required|string|max:100',
            'ifsc_code'       => 'required|string|max:100',
            'balance'         => 'required|numeric|min:0',
        ]);

        $bankAccount->update($data);

        return redirect()->route('bank-accounts.index')->with('success', 'Bank Account updated successfully!');
    }

    public function destroy($id)
    {
        $bankAccount = BankAccount::findOrFail($id);
        $bankAccount->delete();

        return redirect()->route('bank-accounts.index')->with('success', 'Bank Account deleted successfully!');
    }
    
}
