<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Expense;
use App\Models\BankAccount;
use App\Models\BankAccountLedger;

class ExpenseController extends Controller
{
    public function index(Request $request)
    {
        $query = Expense::query();

        if ($search = $request->get('search')) {
            $query->where('expense_date', 'like', "%$search%")
                ->orWhere('category', 'like', "%$search%")
                ->orWhere('amount', 'like', "%$search%");
        }

        $expenses = $query->orderBy('id', 'desc')->paginate(10)->through(function ($inv) {
            return [
                'id' => $inv->id,
                'expense_date' => $inv->expense_date,
                'category'     => $inv->category,
                'amount'       => $inv->amount,
                'description'  => $inv->description,
            ];
        });

        return inertia('Expenses/Index', [
            'expenses' => $expenses,
        ]);
    }

    public function create()
    {
        return Inertia::render('Expenses/Create', [
            'bankAccount' => BankAccount::select('id', 'account_name', 'account_number')->get(),
        ]);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'expense_date'   => 'required|date',
            'category'       => 'required|string|max:100',
            'amount'         => 'required|numeric|min:0',
            'bank_account_id'    => 'required|numeric|min:0',
        ]);

        $data['description'] = $request->input('description') ?? '';

        $expense = Expense::create($data);

        $bankAccount = BankAccount::findOrFail($request->bank_account_id);

        $bankAccount['balance'] -= $request->amount;
        $bankAccount->save();

        $bankAccountLedger = new BankAccountLedger();

        $bankAccountLedger->bank_account_id = $request->bank_account_id;
        $bankAccountLedger->transaction_date = date('Y-m-d');
        $bankAccountLedger->reference = "Expense " . $expense->id;
        $bankAccountLedger->type = "credit";
        $bankAccountLedger->amount = $request->amount;
        $bankAccountLedger->carrier_id = 0;
        $bankAccountLedger->client_id = 0;
        // $bankAccountLedger->description = 'New Payme';

        $bankAccountLedger->save();

        return redirect()->route('expenses.index')->with('success', 'Expense created successfully!');
    }

    public function show($id)
    {
        $expense = Expense::findOrFail($id);

        return Inertia::render('Expenses/Show', [
            'expense' => $expense,
        ]);
    }

    public function edit(string $id)
    {
        $expense = Expense::findOrFail($id);
        return Inertia::render('Expenses/Edit', [
            'expense' => $expense,
        ]);
    }

    public function update(Request $request, string $id)
    {
        $data = $request->validate([
            'expense_date'    => 'required|date',
            'category'         => 'required|string|max:100',
            'amount'         => 'required|numeric|min:0',
        ]);

        $data['description'] = $request->input('description') ?? '';

        $expense = Expense::findOrFail($id);
        $expense->update($data);

        return redirect()->route('expenses.index')->with('success', 'Expense created successfully!');
    }

     public function destroy(string $id)
    {
        $expense = Expense::findOrFail($id);
        $expense->delete();
        return redirect()->route('expenses.index')->with('success', 'Invoice deleted successfully.');
    }
}
