<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\FundTransfer;
use App\Models\BankAccount;
use App\Models\BankAccountLedger;

use Illuminate\Support\Facades\DB;

class FundTransferController extends Controller
{
    public function index(Request $request)
    {
        $query = FundTransfer::query()
            ->with(['fromAccount:id,account_name', 'toAccount:id,account_name']);

        if ($search = $request->get('search')) {
            $query->where('transfer_date', 'like', "%$search%")
                ->orWhereHas('fromAccount', fn($q) => $q->where('account_name', 'like', "%$search%"))
                ->orWhereHas('toAccount', fn($q) => $q->where('account_name', 'like', "%$search%"));
        }

        $fundTransfers = $query->orderBy('id', 'desc')->paginate(10)->through(function ($ft) {
            return [
                'id' => $ft->id,
                'transfer_date'     => $ft->transfer_date,
                'from_account_name' => $ft->fromAccount->account_name ?? '',
                'to_account_name'   => $ft->toAccount->account_name ?? '',
                'amount'            => $ft->amount,
                'status'            => $ft->status,
                'description'       => $ft->description,
            ];
        });

        return inertia('FundTransfers/Index', [
            'fundTransfers' => $fundTransfers,
        ]);
    } 
    
    public function create()
    {
        $bankAccounts = BankAccount::orderBy('account_name')->get(['id', 'account_name', 'balance', 'account_number']);
        return Inertia::render('FundTransfers/Create', [
            'bankAccounts' => $bankAccounts,
        ]);
    }

    public function store(Request $request)
    {
        // ✅ Step 1: Validate Input
        $validated = $request->validate([
            'from_account_id' => 'required|different:to_account_id|exists:bank_accounts,id',
            'to_account_id'   => 'required|exists:bank_accounts,id',
            'amount'          => 'required|numeric|min:0.01',
            'description'     => 'nullable|string|max:255',
        ]);

        DB::beginTransaction();
        try {
            // ✅ Step 2: Fetch Accounts
            $fromAccount = BankAccount::findOrFail($validated['from_account_id']);
            $toAccount   = BankAccount::findOrFail($validated['to_account_id']);

            // ✅ Step 3: Check Balance
            if ($fromAccount->balance < $validated['amount']) {
                return back()->withErrors(['amount' => 'Insufficient balance in the source account.']);
            }

            // ✅ Step 4: Create Fund Transfer Record
            $transfer = FundTransfer::create([
                'from_account_id' => $validated['from_account_id'],
                'to_account_id'   => $validated['to_account_id'],
                'transfer_date'   => now(),
                'amount'          => $validated['amount'],
                'status'          => 'completed', // or 'pending'
                'description'     => $validated['description'] ?? null,
                'created_by'      => auth()->id(),
            ]);

            // ✅ Step 5: Update Balances
            $fromAccount->decrement('balance', $validated['amount']);
            $toAccount->increment('balance', $validated['amount']);

            $bankAccountLedger = new BankAccountLedger();
            $bankAccountLedger->bank_account_id = $validated['from_account_id'];
            $bankAccountLedger->transaction_date = date('Y-m-d');
            $bankAccountLedger->reference = "Fund Transfer to " . $toAccount->account_name;
            $bankAccountLedger->type = "credit";
            $bankAccountLedger->amount = $validated['amount'];
            $bankAccountLedger->carrier_id = 0;
            $bankAccountLedger->client_id = 0;

            $bankAccountLedger->save();

            $bankAccountLedger = new BankAccountLedger();
            $bankAccountLedger->bank_account_id = $validated['to_account_id'];
            $bankAccountLedger->transaction_date = date('Y-m-d');
            $bankAccountLedger->reference = "Fund Transfer from " . $fromAccount->account_name;
            $bankAccountLedger->type = "debit";
            $bankAccountLedger->amount = $validated['amount'];
            $bankAccountLedger->carrier_id = 0;
            $bankAccountLedger->client_id = 0;
            // $bankAccountLedger->description = 'New Payme';

            $bankAccountLedger->save();

            DB::commit();

            return redirect()->route('fund-transfers.index')->with('success', 'Fund transfer completed successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Transfer failed: ' . $e->getMessage()]);
        }
    }

    public function show($id)
    {
        $fundTransfer = FundTransfer::with(['fromAccount:id,account_name,account_number', 'toAccount:id,account_name,account_number'])->findOrFail($id);
        // dd($fundTransfer->toArray());
        return Inertia::render('FundTransfers/Show', [
            'fundTransfer' => [
                'id'                => $fundTransfer->id,
                'transfer_date'     => $fundTransfer->transfer_date,
                'from_account'      => $fundTransfer->fromAccount ? [
                    'id'            => $fundTransfer->fromAccount->id,
                    'account_name'  => $fundTransfer->fromAccount->account_name,
                    'account_number'=> $fundTransfer->fromAccount->account_number,
                ] : null,
                'to_account'        => $fundTransfer->toAccount ? [
                    'id'            => $fundTransfer->toAccount->id,
                    'account_name'  => $fundTransfer->toAccount->account_name,
                    'account_number'=> $fundTransfer->toAccount->account_number,
                ] : null,
                'amount'            => $fundTransfer->amount,
                'status'            => $fundTransfer->status,
                'description'       => $fundTransfer->description,
            ],
        ]);
    }

    public function edit($id)
    {
        $fundTransfer = FundTransfer::findOrFail($id);
        $bankAccounts = BankAccount::orderBy('account_name')->get(['id', 'account_name', 'balance', 'account_number']);

        return Inertia::render('FundTransfers/Edit', [
            'fundTransfer' => [
                'id'                => $fundTransfer->id,
                'from_account_id'   => $fundTransfer->from_account_id,
                'to_account_id'     => $fundTransfer->to_account_id,
                'amount'            => $fundTransfer->amount,
                'description'       => $fundTransfer->description,
            ],
            'bankAccounts' => $bankAccounts,
        ]);
    }

    public function update(Request $request, $id)
    {
        $fundTransfer = FundTransfer::findOrFail($id);

        $validated = $request->validate([
            'from_account_id' => 'required|different:to_account_id|exists:bank_accounts,id',
            'to_account_id'   => 'required|exists:bank_accounts,id',
            'amount'          => 'required|numeric|min:0.01',
            'description'     => 'nullable|string|max:255',
        ]);

        DB::beginTransaction();
        try {
            // Fetch Accounts
            $fromAccount = BankAccount::findOrFail($validated['from_account_id']);
            $toAccount   = BankAccount::findOrFail($validated['to_account_id']);

            // Revert Previous Transfer if Accounts Changed or Amount Changed
            if ($fundTransfer->from_account_id != $validated['from_account_id'] || 
                $fundTransfer->to_account_id != $validated['to_account_id'] || 
                $fundTransfer->amount != $validated['amount']) {

                // Revert previous transfer
                $prevFromAccount = BankAccount::findOrFail($fundTransfer->from_account_id);
                $prevToAccount   = BankAccount::findOrFail($fundTransfer->to_account_id);

                $prevFromAccount->increment('balance', $fundTransfer->amount);
                $prevToAccount->decrement('balance', $fundTransfer->amount);

                // Check Balance for new transfer
                if ($fromAccount->balance < $validated['amount']) {
                    return back()->withErrors(['amount' => 'Insufficient balance in the source account.']);
                }

                // Apply new transfer
                $fromAccount->decrement('balance', $validated['amount']);
                $toAccount->increment('balance', $validated['amount']);
            }

            // Update Fund Transfer Record
            $fundTransfer->update([
                'from_account_id' => $validated['from_account_id'],
                'to_account_id'   => $validated['to_account_id'],
                'amount'          => $validated['amount'],
                'description'     => $validated['description'] ?? null,
                'updated_by'      => auth()->id(),
            ]);

            DB::commit();

            return redirect()->route('fund-transfers.index')->with('success', 'Fund transfer updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Update failed: ' . $e->getMessage()]);
        }   
    }

    public function destroy($id)
    {
        $fundTransfer = FundTransfer::findOrFail($id);

        DB::beginTransaction();
        try {
            // Revert the transfer
            $fromAccount = BankAccount::findOrFail($fundTransfer->from_account_id);
            $toAccount   = BankAccount::findOrFail($fundTransfer->to_account_id);

            $fromAccount->increment('balance', $fundTransfer->amount);
            $toAccount->decrement('balance', $fundTransfer->amount);

            // Delete the fund transfer record
            $fundTransfer->delete();

            DB::commit();

            return redirect()->route('fund-transfers.index')->with('success', 'Fund transfer deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Deletion failed: ' . $e->getMessage()]);
        }
    }   
}
