<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Client;
use App\Models\Company;
use App\Models\HotelBooking;
use App\Models\BankAccount;


class HotelBookingController extends Controller
{
    public function index(Request $request)
    {
        $query = HotelBooking::query()
            ->with(['client:id,name', 'company:id,name']);

        if ($search = $request->get('search')) {
            $query->where('date', 'like', "%$search%")
                ->orWhereHas('client', fn($q) => $q->where('name', 'like', "%$search%"))
                ->orWhereHas('company', fn($q) => $q->where('name', 'like', "%$search%"));
        }

        $hotel_bookings = $query->orderBy('id', 'desc')->paginate(10)->through(function ($inv) {
            return [
                'id' => $inv->id,
                'date'  => $inv->date,
                'type'   => $inv->type,
                'client_name'   => $inv->client->name ?? '',
                'company_name'  => $inv->company->name ?? '',
                'date_from'   => $inv->date_from,
                'date_to'     => $inv->date_to,
                'base_fare'     => $inv->base_fare,
                'tax'           => $inv->tax,
                'mark_up'       => $inv->mark_up,
                'purchase_com'  => $inv->purchase_com,
                'sales_com'     => $inv->sales_com,
                'discount_lvl'  => $inv->discount_lvl,
                'sales_amount'  => $inv->sales_amount,
                'purchase_amount'=> $inv->purchase_amount,
                'profit_loss'     => $inv->profit_loss,
                'note'             => $inv->note,
            ];
        });

        return inertia('Hotel Booking/Index', [
            'hotel_bookings' => $hotel_bookings,
        ]);
    }

    public function create()
    {
        return Inertia::render('Hotel Booking/Create', [
            'clients' => Client::select('id', 'name', 'passport_number', 'address', 'phone')->get(),
            'companies' => Company::select('id', 'name')->get(),
            'bankAccount' => BankAccount::select('id', 'account_name', 'account_number')->get(),
        ]);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'type'      => 'required|string|max:255',
            'bank_account_id'   => 'required',
            'client_id'      => 'required|exists:clients,id',
            'company_id'     => 'required|exists:companies,id',
            'date_from'    => 'required|date',
            'date_to'    => 'required|date',
            'base_fare'      => 'required|numeric|min:0',
            'tax'            => 'nullable|numeric|min:0',
            'mark_up'        => 'nullable|numeric|min:0',
            'purchase_com'   => 'nullable|numeric|min:0',
            'sales_com'      => 'nullable|numeric|min:0',
            'discount_lvl'   => 'nullable|numeric|min:0',
            'sales_amount'   => 'required|numeric|min:0',
            'purchase_amount'=> 'required|numeric|min:0',
            'profit_loss'    => 'required|numeric',
            'note'            => 'nullable|string|max:100',
        ]);

        $data['date'] = date('Y-m-d');
        $data['created_by'] = auth()->id();
        $data['updated_by'] = auth()->id();

        HotelBooking::create($data);

        return redirect()->route('hotel-booking.index')->with('success', 'Booking created successfully!');
    }

    public function show($id)
    {
        $booking = HotelBooking::with(['client:id,name', 'company:id,name', 'createdBy:id,name', 'updatedBy:id,name',])->findOrFail($id);

        return Inertia::render('Hotel Booking/Show', [
            'booking' => $booking,
        ]);
    }

    public function edit(string $id)
    {
        $booking = HotelBooking::with(['client:id,name', 'company:id,name', 'createdBy:id,name', 'updatedBy:id,name',])->findOrFail($id);
        return Inertia::render('Hotel Booking/Edit', [
            'clients' => Client::select('id', 'name', 'passport_number', 'address', 'phone')->get(),
            'companies' => Company::select('id', 'name')->get(),
            'booking' => $booking,
            // 'purchaseOrders' => $purchaseOrders,
        ]);
    }

    public function update(Request $request, string $id)
    {
        $data = $request->validate([
            'type'      => 'required|string|max:255',
            'client_id'      => 'required|exists:clients,id',
            'company_id'     => 'required|exists:companies,id',
            'date_from'    => 'required|date',
            'date_to'    => 'required|date',
            'base_fare'      => 'required|numeric|min:0',
            'tax'            => 'nullable|numeric|min:0',
            'mark_up'        => 'nullable|numeric|min:0',
            'purchase_com'   => 'nullable|numeric|min:0',
            'sales_com'      => 'nullable|numeric|min:0',
            'discount_lvl'   => 'nullable|numeric|min:0',
            'sales_amount'   => 'required|numeric|min:0',
            'purchase_amount'=> 'required|numeric|min:0',
            'profit_loss'    => 'required|numeric',
            'note'            => 'nullable|string|max:100',
        ]);

        $data['updated_by'] = auth()->id();

        $booking = HotelBooking::findOrFail($id);
        $booking->update($data);

        return redirect()->route('hotel-booking.index')->with('success', 'Invoice created successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $booking = HotelBooking::findOrFail($id);
        $booking->delete();
        return redirect()->route('hotel-booking.index')->with('success', 'Booking deleted successfully.');
    }

}