<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Income;

class IncomeController extends Controller
{
    public function index(Request $request)
    {
        $query = Income::query();

        if ($search = $request->get('search')) {
            $query->where('income_date', 'like', "%$search%")
                ->orWhere('source', 'like', "%$search%")
                ->orWhere('amount', 'like', "%$search%");
        }

        $incomes = $query->orderBy('id', 'desc')->paginate(10)->through(function ($inv) {
            return [
                'id' => $inv->id,
                'income_date'  => $inv->income_date,
                'source'       => $inv->source,
                'amount'       => $inv->amount,
                'description'  => $inv->description,
            ];
        });

        return inertia('Incomes/Index', [
            'incomes' => $incomes,
        ]);
    }

    public function create()
    {
        return Inertia::render('Incomes/Create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'income_date'    => 'required|date',
            'source'         => 'required|string|max:100',
            'amount'         => 'required|numeric|min:0',
        ]);

        $data['description'] = $request->input('description') ?? '';

        Income::create($data);

        return redirect()->route('incomes.index')->with('success', 'Income created successfully!');
    }

    public function show($id)
    {
        $income = Income::findOrFail($id);

        return Inertia::render('Incomes/Show', [
            'income' => $income,
        ]);
    }

    public function edit(string $id)
    {
        $income = Income::findOrFail($id);
        return Inertia::render('Incomes/Edit', [
            'income' => $income,
        ]);
    }

    public function update(Request $request, string $id)
    {
        $data = $request->validate([
            'income_date'    => 'required|date',
            'source'         => 'required|string|max:100',
            'amount'         => 'required|numeric|min:0',
        ]);

        $data['description'] = $request->input('description') ?? '';

        $income = Income::findOrFail($id);
        $income->update($data);

        return redirect()->route('incomes.index')->with('success', 'Income created successfully!');
    }

     public function destroy(string $id)
    {
        $income = Income::findOrFail($id);
        $income->delete();
        return redirect()->route('incomes.index')->with('success', 'Invoice deleted successfully.');
    }

}
