<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Client;
use App\Models\Company;
use App\Models\Invoice;
use App\Models\Carrier;
use App\Models\PurchaseOrder;
use App\Models\Payment;
use App\Models\PaymentDetail;
use App\Models\BankAccount;
use App\Models\BankAccountLedger;
use Mpdf\Mpdf;
use Illuminate\Support\Facades\DB;

use Barryvdh\DomPDF\Facade\Pdf;

class InvoiceController extends Controller
{
    public function index(Request $request)
    {
        $query = Invoice::query()
            ->with(['client:id,name', 'company:id,name', 'carrier']);

        if ($search = $request->get('search')) {
            $query->where('invoice_date', 'like', "%$search%")
                ->orWhere('ticket_no', 'like', "%$search%")
                ->orWhereHas('client', fn($q) => $q->where('name', 'like', "%$search%"))
                ->orWhereHas('carrier', fn($q) => $q->where('name', 'like', "%$search%"))
                ->orWhereHas('company', fn($q) => $q->where('name', 'like', "%$search%"));
        }

        $invoices = $query->orderBy('id', 'desc')->paginate(10)->through(function ($inv) {
            return [
                'id' => $inv->id,
                'invoice_date'  => $inv->invoice_date,
                'ticket_no'     => $inv->ticket_no,
                'client_name'   => $inv->client->name ?? '',
                'company_name'  => $inv->company->name ?? '',
                'carrier_name'  => $inv->carrier->name ?? '',
                'ticket_type'   => $inv->ticket_type,
                'flight_date'   => $inv->flight_date,
                'base_fare'     => $inv->base_fare,
                'tax'           => $inv->tax,
                'mark_up'       => $inv->mark_up,
                'purchase_com'  => $inv->purchase_com,
                'sales_com'     => $inv->sales_com,
                'discount_lvl'  => $inv->discount_lvl,
                'sales_amount'  => $inv->sales_amount,
                'purchase_amount'=> $inv->purchase_amount,
                'profit_loss'     => $inv->profit_loss,
                'pnr'             => $inv->pnr,
            ];
        });

        return inertia('Invoices/Index', [
            'invoices' => $invoices,
        ]);
    }

    public function create()
    {
        // $purchaseOrders = PurchaseOrder::select('id', 'order_number')->orderBy('order_number')->get();
        return Inertia::render('Invoices/Create', [
            'clients' => Client::select('id', 'name', 'passport_number', 'address', 'phone')->get(),
            'companies' => Company::select('id', 'name')->get(),
            'carriers' => Carrier::select('id', 'name')->get(),
            // 'purchaseOrders' => $purchaseOrders,
        ]);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'ticket_no'      => 'required|string|max:255',
            'client_id'      => 'required|exists:clients,id',
            'company_id'     => 'required|exists:companies,id',
            'carrier_id'     => 'required|exists:carriers,id',
            'ticket_type'    => 'required|string|max:100',
            'flight_date'    => 'required|date',
            'base_fare'      => 'required|numeric|min:0',
            'tax'            => 'nullable|numeric|min:0',
            'mark_up'        => 'nullable|numeric|min:0',
            'purchase_com'   => 'nullable|numeric|min:0',
            'sales_com'      => 'nullable|numeric|min:0',
            'discount_lvl'   => 'nullable|numeric|min:0',
            'sales_amount'   => 'required|numeric|min:0',
            'purchase_amount'=> 'required|numeric|min:0',
            'profit_loss'    => 'required|numeric',
            'pnr'            => 'nullable|string|max:100',
        ]);

        $data['invoice_date'] = date('Y-m-d');
        $data['created_by'] = auth()->id();
        $data['updated_by'] = auth()->id();

        $invoice = Invoice::create($data);

        $payment = new Payment();
        // $payment_details = new PaymentDetail();
        $payment->invoice_id = $invoice->id;
        $payment->client_id = $invoice->client_id;
        $payment->total_amount = $invoice->sales_amount;

        $payment->paid_amount = 0;
        $payment->date = date('Y-m-d');

        $payment->due_amount = $request->sales_amount;

        $payment->save();

        return redirect()->route('invoices.index')->with('success', 'Invoice created successfully!');
    }

    public function show($id)
    {
        $invoice = Invoice::with(['client:id,name', 'company:id,name', 'carrier:id,name', 'createdBy:id,name', 'updatedBy:id,name',])->findOrFail($id);

        return Inertia::render('Invoices/Show', [
            'invoice' => $invoice,
        ]);
    }

    public function edit(string $id)
    {
        $invoice = Invoice::with(['client:id,name', 'company:id,name', 'carrier:id,name', 'createdBy:id,name', 'updatedBy:id,name',])->findOrFail($id);
        return Inertia::render('Invoices/Edit', [
            'clients' => Client::select('id', 'name', 'passport_number', 'address', 'phone')->get(),
            'companies' => Company::select('id', 'name')->get(),
            'carriers' => Carrier::select('id', 'name')->get(),
            'invoice' => $invoice,
            // 'purchaseOrders' => $purchaseOrders,
        ]);
    }

    public function update(Request $request, string $id)
    {
        $data = $request->validate([
            'invoice_date'   => 'required|date',
            'ticket_no'      => 'required|string|max:255',
            'client_id'      => 'required|exists:clients,id',
            'company_id'     => 'required|exists:companies,id',
            'carrier_id'     => 'required|exists:carriers,id',
            'ticket_type'    => 'required|string|max:100',
            'flight_date'    => 'required|date',
            'base_fare'      => 'required|numeric|min:0',
            'tax'            => 'nullable|numeric|min:0',
            'mark_up'        => 'nullable|numeric|min:0',
            'purchase_com'   => 'nullable|numeric|min:0',
            'sales_com'      => 'nullable|numeric|min:0',
            'discount_lvl'   => 'nullable|numeric|min:0',
            'sales_amount'   => 'required|numeric|min:0',
            'purchase_amount'=> 'required|numeric|min:0',
            'profit_loss'    => 'required|numeric',
            'pnr'            => 'nullable|string|max:100',
        ]);

        $data['updated_by'] = auth()->id();

        $invoice = Invoice::findOrFail($id);

        // $oldPaidAmount = $invoice->paid_amount;
        // $oldCarrierId = $invoice->carrier_id;

        $invoice->update($data);

        // $payment = Payment::updateOrCreate(
        //     ['invoice_id' => $invoice->id],
        //     [
        //         'client_id' => $invoice->client_id,
        //         'total_amount' => $invoice->sales_amount,
        //         'paid_amount' => $request->paid_amount,
        //         'due_amount' => $invoice->sales_amount - $request->paid_amount,
        //         'date' => $request->invoice_date,
        //     ]
        // );

        // $payment_details = PaymentDetails::where('invoice_id', $invoice->id)
        //     ->orderBy('id', 'desc')
        //     ->limit(1)
        //     ->first();


        // if($oldCarrierId == $request->carrier_id) {
        //     $bankAccountId = Carrier::where('id', $request->carrier_id)->value('bank_account_id');
        //     $bankAccount = BankAccount::findOrFail($bankAccountId);
        //     $bankAccount->balance -= $oldPaidAmount;
        //     $bankAccount->balance += $request->paid_amount;
        // } else {
        //     // new carrier
        //     $bankAccountId = Carrier::where('id', $request->carrier_id)->value('bank_account_id');
        //     $bankAccount = BankAccount::findOrFail($bankAccountId);
        //     $bankAccount->balance += $request->paid_amount;
        //     $bankAccount->save();

        //     // old carrier balance correction
        //     $oldBankAccountID = Carrier::where('id', $oldCarrierId)->value('bank_account_id');
        //     $oldBankAccount = BankAccount::findOrFail($oldBankAccountID);
        //     $oldBankAccount->balance -= $oldPaidAmount;
        //     $oldBankAccount->save();
        // }

        // // --------------------------
        // // 📘 Update Ledger Entry
        // // --------------------------
        // BankAccountLedger::updateOrCreate(
        //     [
        //         'bank_account_id' => $bankAccount->id,
        //         'reference' => 'Invoice Id ' . $invoice->id,
        //     ],
        //     [
        //         'transaction_date' => $request->invoice_date,
        //         'type' => 'debit',
        //         'amount' => $request->paid_amount,
        //         'carrier_id' => $request->carrier_id,
        //         'client_id' => $invoice->client_id,
        //         'description' => 'Invoice updated and payment adjusted',
        //     ]
        // );

        // // If carrier changed, adjust old ledger too
        // if ($oldCarrierId != $request->carrier_id) {
        //     BankAccountLedger::updateOrCreate(
        //         [
        //             'bank_account_id' => $oldBankAccount->id,
        //             'reference' => 'Invoice Id ' . $invoice->id . ' (Old Carrier)',
        //         ],
        //         [
        //             'transaction_date' => $request->invoice_date,
        //             'type' => 'credit',
        //             'amount' => $oldPaidAmount,
        //             'carrier_id' => $oldCarrierId,
        //             'client_id' => $invoice->client_id,
        //             'description' => 'Carrier changed — reverse old payment',
        //         ]
        //     );
        // }
        
        return redirect()->route('invoices.index')->with('success', 'Invoice created successfully!');
    }

    
    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $invoice = Invoice::findOrFail($id);
        $invoice->delete();
        return redirect()->route('invoices.index')->with('success', 'Invoice deleted successfully.');
    }

    public function print($id)
    {
        $invoice = Invoice::with(['client', 'company', 'payment'])->findOrFail($id);
        // $invoice = Invoice::with('client')->findOrFail($id);
        $html = view('invoices.pdf', compact('invoice'))->render();

        $mpdf = new Mpdf([
            'default_font_size' => 12,
            'default_font' => 'dejavusans',
            'format' => 'A4',
        ]);

        // 🖨 Add JS to auto-open print dialog
        $mpdf->SetJS('this.print();');
        $mpdf->WriteHTML($html);

        return response($mpdf->Output("invoice-{$invoice->id}.pdf", 'I'))
            ->header('Content-Type', 'application/pdf');
    }

    public function printList()
    {
        $invoice = Invoice::with(['client', 'carrier', 'payment'])->OrderBy('id', 'desc')->get();
        // dd($invoice->toArray());
        return Inertia::render('Invoices/PrintList', [
            'invoice' => $invoice,
            // 'purchaseOrders' => $purchaseOrders,
        ]);
    }

    public function printMultiple(Request $request)
    {
        $ticketIds = json_decode($request->tickets, true);

        if (!$ticketIds || !is_array($ticketIds) || count($ticketIds) === 0) {
            abort(400, 'No tickets selected');
        }

        // Fetch invoices with relations
        $invoices = Invoice::with(['client', 'carrier', 'payment'])
            ->whereIn('id', $ticketIds)
            ->orderBy('id', 'asc')
            ->get();

            // dd($invoices->toArray());

        // Load view
        $pdf = Pdf::loadView('invoices.pdf-multiple', [
            'invoices' => $invoices
        ]);

        // Stream PDF to browser (open in new tab)
        return $pdf->stream('invoices.pdf');
    }
}