<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\PurchaseOrder;
use App\Models\Vendor;
use Inertia\Inertia;

class PurchaseOrderController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = PurchaseOrder::query()->with(['vendor:id,name']);
        if ($search = $request->input('search')) {
            $query->where(function($q) use ($search) {
                $q->where('order_number', 'like', "%$search%")
                  ->orWhereHas('vendor', function($v) use ($search) {
                      $v->where('name', 'like', "%$search%") ;
                  })
                  ->orWhere('order_date', 'like', "%$search%")
                  ->orWhere('total_amount', 'like', "%$search%")
                  ->orWhere('paid_amount', 'like', "%$search%")
                  ->orWhere('due_amount', 'like', "%$search%")
                  ->orWhere('paid_via', 'like', "%$search%")
                  ->orWhere('status', 'like', "%$search%")
                  ->orWhere('notes', 'like', "%$search%") ;
            });
        }
        $orders = $query->orderByDesc('id')->paginate(15)->withQueryString();
        return Inertia::render('App Settings/Purchase Orders/Index', [
            'orders' => $orders,
            'query' => $request->input('search', ''),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $vendors = Vendor::select('id', 'name')->orderBy('name')->get();
        return Inertia::render('App Settings/Purchase Orders/Create', [
            'vendors' => $vendors
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'order_number' => 'required|string|max:255',
            'vendor_id' => 'required|exists:vendors,id',
            'order_date' => 'required|date',
            'total_amount' => 'required|numeric|min:0',
            'paid_amount' => 'numeric|min:0',
            'due_amount' => 'numeric|min:0',
            'paid_via' => 'nullable|string|max:100',
            'status' => 'nullable|string|max:50',
            'notes' => 'nullable|string|max:1000',
        ]);
        PurchaseOrder::create($validated);
        return redirect()->route('purchase-orders.index')->with('success', 'Purchase Order created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $order = PurchaseOrder::with('vendor:id,name')->findOrFail($id);
        return Inertia::render('App Settings/Purchase Orders/Show', [
            'order' => $order
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $order = PurchaseOrder::with('vendor:id,name')->findOrFail($id);
        $vendors = Vendor::select('id', 'name')->orderBy('name')->get();
        return Inertia::render('App Settings/Purchase Orders/Edit', [
            'order' => $order,
            'vendors' => $vendors
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validated = $request->validate([
            'order_number' => 'required|string|max:255',
            'vendor_id' => 'required|exists:vendors,id',
            'order_date' => 'required|date',
            'total_amount' => 'required|numeric|min:0',
            'paid_amount' => 'numeric|min:0',
            'due_amount' => 'numeric|min:0',
            'paid_via' => 'nullable|string|max:100',
            'status' => 'string|max:50',
            'notes' => 'nullable|string|max:1000',
        ]);
        $order = PurchaseOrder::findOrFail($id);
        $order->update($validated);
        return redirect()->route('purchase-orders.index')->with('success', 'Purchase Order updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $order = PurchaseOrder::findOrFail($id);
        $order->delete();
        return redirect()->route('purchase-orders.index')->with('success', 'Purchase Order deleted successfully.');
    }
}
