<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use Inertia\Inertia;

use App\Models\Refund;
use App\Models\Invoice;
use App\Models\BankAccount;
use App\Models\BankAccountLedger;

use Barryvdh\DomPDF\Facade\Pdf;
use Mpdf\Mpdf;

class RefundController extends Controller
{
    public function index(Request $request)
    {
        $query = Refund::query();

        if ($search = $request->get('search')) {
            $query->where('ticket_no', 'like', "%$search%")
                ->orWhere('refund_date', 'like', "%$search%");
        }

        $refunds = $query->orderBy('id', 'desc')->paginate(10)->through(function ($inv) {
            return [
                'id' => $inv->id,
                'refund_date'  => $inv->refund_date,
                'ticket_no'     => $inv->ticket_no,
                'refund_type'   => $inv->refund_type,
                'sales_amount'  => $inv->sales_amount,
                'purchase_amount'  => $inv->purchase_amount,
                'refundable_tax' => $inv->refundable_tax,
                'non_refundable' => $inv->non_refundable,
                'cancellation_fee' => $inv->cancellation_fee,
                'admin_fee'        => $inv->admin_fee,
                'carrier_refund_amount'    => $inv->carrier_refund_amount,
                'refund_amount'    => $inv->refund_amount,                
            ];
        });

        return inertia('Refunds/Index', [
            'refunds' => $refunds,
        ]);
    }

    public function create()
    {
        return Inertia::render('Refunds/Create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'ticket_no' => 'required|string|max:255',
            'refund_type' => 'required|string|max:255',
            'base_fare' => 'required|numeric|min:0',
            'tax' => 'required|numeric|min:0',
            'gross_fare' => 'required|numeric|min:0',
            'mark_up' => 'nullable|numeric|min:0',
            'discount_lvl' => 'nullable|numeric|min:0',
            'net_fare' => 'required|numeric|min:0',
            'refundable_tax' => 'nullable|numeric|min:0',
            'non_refundable' => 'nullable|numeric|min:0',
            'cancellation_fee' => 'nullable|numeric|min:0',
            'admin_fee' => 'nullable|numeric|min:0',
            'refund_amount' => 'required|numeric|min:0',
        ]);

        $validated['refund_date'] = date('Y-m-d');
        $validated['created_by'] = Auth()->id();

        Refund::create($validated);

        return redirect()->route('refunds.index')->with('success', 'Refund created successfully.');
    }   

    public function show($id)
    {
        $refund = Refund::findOrFail($id);

        return Inertia::render('Refunds/Show', [
            'refund' => $refund,
        ]);
    }   

    public function edit($id)
    {
        $refund = Invoice::findOrFail($id);
        // dd($refund);
        return Inertia::render('Refunds/Edit', [
            'refund' => $refund,
        ]);
    } 
    
    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'invoice_id'        => 'required|exists:invoices,id',
            'ticket_no'         => 'required|string|max:255',
            'refund_type'       => 'required|string|max:255',
            'sales_amount'      => 'nullable|numeric|min:0',
            'purchase_amount'   => 'nullable|numeric|min:0',
            'refundable_tax'    => 'nullable|numeric|min:0',
            'non_refundable'    => 'nullable|numeric|min:0',
            'cancellation_fee'  => 'nullable|numeric|min:0',
            'admin_fee'         => 'nullable|numeric|min:0',
            'refund_amount'     => 'required|numeric|min:0',
            'carrier_refund_amount'   => 'required|numeric|min:0',
        ]);

        $validated['refund_date'] = now();
        $validated['updated_by']  = auth()->id();

        // 🔹 If refund exists, update. Otherwise, create.
        $refund = Refund::updateOrCreate(
            ['id' => $id],  // condition (find existing refund)
            $validated      // new or updated data
        );

        // Optional: Update related invoice balance or ledger here
        // Example:
        // $invoice = Invoice::find($validated['invoice_id']);
        // $invoice->update(['status' => 'Refunded']);

        $invoice = Invoice::with('carrier')->findOrFail($request->invoice_id);

        $bankAccountId = $invoice->carrier->bank_account_id;

        $bankAccount = BankAccount::findOrFail($bankAccountId);

        $bankAccount['balance'] += $request->carrier_refund_amount;
        $bankAccount['balance'] -= $request->refund_amount;
        $bankAccount->save();

        $bankAccountLedger = new BankAccountLedger();

        $bankAccountLedger->bank_account_id = $bankAccountId;
        $bankAccountLedger->transaction_date = date('Y-m-d');
        $bankAccountLedger->reference = "Refund from Carrier Invoice Id " . $request->invoice_id;
        $bankAccountLedger->type = "debit";
        $bankAccountLedger->amount = $request->carrier_refund_amount;
        $bankAccountLedger->carrier_id = $invoice->carrier_id;
        $bankAccountLedger->client_id = $invoice->client_id;
        // $bankAccountLedger->description = 'New Payme';

        $bankAccountLedger->save();

        $bankAccountLedger = new BankAccountLedger();

        $bankAccountLedger->bank_account_id = $bankAccountId;
        $bankAccountLedger->transaction_date = date('Y-m-d');
        $bankAccountLedger->reference = "Refund to Client Invoice Id " . $request->invoice_id;
        $bankAccountLedger->type = "credit";
        $bankAccountLedger->amount = $request->refund_amount;
        $bankAccountLedger->carrier_id = $invoice->carrier_id;
        $bankAccountLedger->client_id = $invoice->client_id;
        // $bankAccountLedger->description = 'New Payme';

        $bankAccountLedger->save();


        return redirect()->route('refunds.index')->with('success', 'Refund saved successfully.');
    }

    
    public function destroy($id)
    {
        $refund = Refund::findOrFail($id);
        $refund->delete();

        return redirect()->route('refunds.index')->with('success', 'Refund deleted successfully.');
    }

    public function print($id) 
    {
        $refund_invoice = Refund::findOrFail($id);

        // dd($refund_invoice);

        // Load view
        $pdf = Pdf::loadView('invoices.refund-pdf', [
            'refund_invoice' => $refund_invoice
        ]);

        // Stream PDF to browser (open in new tab)
        return $pdf->stream('refund-invoice.pdf');
    }
}
