<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\User;
use Illuminate\Support\Facades\Hash;

use Spatie\Permission\Models\Role;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return Inertia::render("Users/Index", [
            "users" => User::with("roles")->get()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render("Users/Create", [
            "roles" => Role::pluck("name")->all()
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name'       => 'required|string|max:20',
            'email'      => 'required|email',
            'password'   => 'required',
        ]);

        $user = User::create(
            $request->only(["name", "email"]) + ["password" => Hash::make($request->password)]
        );

        $user->syncRoles($request->roles);

        return redirect()->route('users.index')
                         ->with('success', 'User created Successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
         return Inertia::render("Users/Show", [
            "user"  => User::findOrFail($id)
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $user = User::findOrFail($id);
        return Inertia::render("Users/Edit", [
            "user"  => $user,
            "userRoles" => $user->roles()->pluck("name")->all(),
            "roles" => Role::pluck("name")->all()
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $request->validate([
            'name'  => 'required|string|max:20',
            'email' => 'required|email',
        ]);

        $user = User::findOrFail($id);

        $user->name  = $request->name;
        $user->email = $request->email;

        if ($request->password) {
            $user->password = Hash::make($request->password);
        }

        $user->save();

        $user->syncRoles($request->roles);

        return redirect()->route('users.index')
                        ->with('success', 'User updated Successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $user = User::findOrFail($id);
        $user->delete();
        return redirect()->route('users.index')
                        ->with('success', 'User deleted Successfully!');
    }
}
